<?php

namespace App\Http\Controllers\Administrator;

use App\Http\Controllers\Controller;
use App\Models\Course;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class CourseController extends Controller
{
    public function index()
    {
        $courses = Course::paginate(10);
        $data = [
            'title' => 'course',
            'sidebar_menu' => 'course',
            'course' => $courses,
        ];
        return view('admin.course.index')->with($data);
    }
    public function all_course(Request $request)
    {
        if ($request->ajax()) {
            return DataTables::of(Course::latest()->get())
                ->addIndexColumn()
                ->addColumn('course_id', function ($data) {
                    return $data->course_id ?? '-';
                })
                // ->addColumn('course_name', function ($data) {
                //     return Str::limit($data->course_name, 20, '...') ?? '---';
                // })
                ->addColumn('course_name', function ($data) {
                    return $data->course_name ?? '';
                })
                ->addColumn('course_price', function ($data) {
                    return get_price($data->course_price);
                })
                ->addColumn('action', function ($data) {
                    return view('admin.course.include.action_td', ['data' => $data]);
                })
                ->filter(function ($query) {
                    if (request()->input('search')) {
                        $query->where(function ($search_query) {
                            $search_query->whereLike(['course_name', 'type'], request()->input('search'));
                        });
                    }
                })
                ->make(true);
        }
        $data = array(

            'breadcrumb' => 'All Ads',
            'title'    => 'All Ads',
        );
        return view('admin.course.index', $data);
    }
    public function add_course()
    {
        $data = array(
            'title' => 'Add Course',
            'sub_title' => 'Add Course',
            'sidebar_menu' => 'course',
        );

        return view('admin.course.add_course')->with($data);
    }
    public function edit($id)
    {
        $data = array(
            'title' => 'Course',
            'edit' => Course::hashidFind($id),
            'sub_title' => 'Edit Course',
            'sidebar_menu' => 'course',

        );
        return view('admin.course.add_course')->with($data);
    }
    public function delete($id)
    {
        $course = Course::hashidFind($id);

        if ($course) {
            $course->delete();

            return response()->json([
                'success' => 'Course deleted successfully.',
                'reload' => true,
            ]);
        } else {
            return response()->json([
                'success' => 'Course not found.',
                'reload' => true,
            ], 404);
        }
    }
    public function save(Request $request)
    {
        $request->validate([
            'course_id' => 'required|numeric',
            'course_name' => 'required|string|max:255',
            'course_price' => 'nullable|numeric',
        ]);

        DB::beginTransaction();

        try {
            if ($request->id) {
                $course = Course::findOrFail($request->id);
                $text = 'Course Updated Successfully';
            } else {
                $course = new Course();
                $text = 'Course Created Successfully';
            }

            $course->course_id = $request->course_id;
            $course->course_name = $request->course_name;
            $course->course_price = $request->course_price;

            $course->save();
            DB::commit();

            if ($request->ajax() && $request->check_next == 1) {
                $msg = [
                    'success' => $text,
                    'redirect' => route('admin.course.add'),
                ];
                return response()->json($msg);
            } else {
                $msg = [
                    'success' => $text,
                    'redirect' => route('admin.course.course'),
                ];
                return response()->json($msg);
            }
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong! Please try again.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
